<?php
// app/models/Visitor.php

class Visitor
{
    private $conn;

    public function __construct($conn)
    {
        $this->conn = $conn;
    }

    public function getAll()
    {
        $sql = "
            SELECT v.*, 
                   h.name AS host_name,
                   d.name AS department_name
            FROM visitors v
            LEFT JOIN hosts h ON v.host_id = h.id
            LEFT JOIN departments d ON v.department_id = d.id
            ORDER BY v.id DESC
        ";

        return $this->conn->query($sql);
    }

    public function find($id)
    {
        $stmt = $this->conn->prepare("
            SELECT v.*, 
                   h.name AS host_name,
                   d.name AS department_name
            FROM visitors v
            LEFT JOIN hosts h ON v.host_id = h.id
            LEFT JOIN departments d ON v.department_id = d.id
            WHERE v.id = ?
            LIMIT 1
        ");

        $stmt->bind_param("i", $id);
        $stmt->execute();

        return $stmt->get_result()->fetch_assoc();
    }
    
    public function checkout($id)
   {
    $sql = "
        UPDATE visitors 
        SET status = 'checked_out',
            checkout_time = NOW()
        WHERE id = ? AND status = 'checked_in'
    ";

    $stmt = $this->conn->prepare($sql);
    $stmt->bind_param("i", $id);
    return $stmt->execute();
    }
    
public function updateStatus($id, $status)
{
    $stmt = $this->conn->prepare("UPDATE visitors SET status = ? WHERE id = ?");
    $stmt->bind_param("si", $status, $id);
    return $stmt->execute();
}

    public function delete($id)
    {
        $stmt = $this->conn->prepare("DELETE FROM visitors WHERE id = ?");
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }
    
    public function getReturningVisitors()
{
    $sql = "
        SELECT v.*
        FROM visitors v
        WHERE v.status = 'checked_out'
           OR v.status = 'expected'
           OR v.status = 'inducted'
        ORDER BY v.id DESC
    ";
    return $this->conn->query($sql);
}

public function getCheckedInVisitors()
{
    $sql = "
        SELECT v.*, h.name AS host_name, d.name AS department_name
        FROM visitors v
        LEFT JOIN hosts h ON v.host_id = h.id
        LEFT JOIN departments d ON v.department_id = d.id
        WHERE v.status = 'checked_in'
        ORDER BY v.checkin_datetime DESC
    ";
    return $this->conn->query($sql);
}

public function createInduction($data)
{
    $sql = "
        INSERT INTO visitors (
            name,
            phone,
            email,
            company,
            department_id,
            induction_datetime,
            status
        ) VALUES (?, ?, ?, ?, ?, ?, ?)
    ";

    $stmt = $this->conn->prepare($sql);

    $stmt->bind_param(
        "ssssiis",
        $data['name'],
        $data['phone'],
        $data['email'],
        $data['company'],
        $data['department_id'],
        $data['induction_datetime'],
        $data['status']
    );

    if ($stmt->execute()) {
        return $this->conn->insert_id;
    }

    return false;
}

public function getInductionRecords($start = null, $end = null)
{
    $sql = "
        SELECT v.*,
               e.name AS host_name,
               d.name AS department_name,
               DATE(v.induction_datetime) AS induction_date
        FROM visitors v
        LEFT JOIN employees  e ON v.host_id = e.id
        LEFT JOIN departments d ON v.department_id = d.id
        WHERE v.status = 'inducted'
    ";

    if ($start && $end) {
        $sql .= " AND DATE(v.induction_datetime) BETWEEN ? AND ? ";
    }

    $sql .= " ORDER BY v.induction_datetime DESC";

    $stmt = $this->conn->prepare($sql);

    if ($start && $end) {
        $stmt->bind_param("ss", $start, $end);
    }

    $stmt->execute();
    return $stmt->get_result();
}

public function getRecentInductions()
{
    $sql = "
        SELECT v.*, d.name AS department_name
        FROM visitors v
        LEFT JOIN departments d ON v.department_id = d.id
        WHERE v.induction_datetime IS NOT NULL
        ORDER BY v.induction_datetime DESC
        LIMIT 50
    ";

    return $this->conn->query($sql);
}

public function create($data)
{
    $sql = "
        INSERT INTO visitors (
            name,
            phone,
            email,
            company,
            gender,
            address,
            purpose,
            items_declared,
            tag_number,
            host_id,
            department_id,
            photo,
            attachment,
            checkin_date,
            checkin_time,
            checkin_datetime,
            status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ";

    $stmt = $this->conn->prepare($sql);

    // Build combined datetime from date + time
    $checkinDatetime = $data['checkin_date'] . ' ' . $data['checkin_time'];

    $stmt->bind_param(
        "sssssssssiissssss",
        $data['name'],            // s
        $data['phone'],           // s
        $data['email'],           // s
        $data['company'],         // s
        $data['gender'],          // s
        $data['address'],         // s
        $data['purpose'],         // s
        $data['items_declared'],  // s
        $data['tag_number'],      // s
        $data['host_id'],         // i
        $data['department_id'],   // i
        $data['photo'],           // s
        $data['attachment'],      // s
        $data['checkin_date'],    // s
        $data['checkin_time'],    // s
        $checkinDatetime,         // s
        $data['status']           // s
    );

    if ($stmt->execute()) {
        return $this->conn->insert_id;
    }

    return false;
}

public function getReports(array $filters = [])
{
    $status = $filters['status'] ?? 'all';
    $start  = $filters['start']  ?? null;
    $end    = $filters['end']    ?? null;
    $search = $filters['search'] ?? null;

    $sql = "
       SELECT v.*,
               e.name AS host_name,
               d.name AS department_name
        FROM visitors v
        LEFT JOIN employees  e ON v.host_id = e.id
        LEFT JOIN departments d ON v.department_id = d.id
        WHERE 1 = 1
    ";

    $types  = '';
    $values = [];

    // Filter by status (if not 'all')
    if ($status !== 'all') {
        $sql    .= " AND v.status = ? ";
        $types  .= 's';
        $values[] = $status;
    }

    // Filter by date range (based on check-in date)
    if (!empty($start) && !empty($end)) {
        $sql    .= " AND DATE(v.checkin_datetime) BETWEEN ? AND ? ";
        $types  .= 'ss';
        $values[] = $start;
        $values[] = $end;
    }

    // Search (name / phone / tag / company)
    if (!empty($search)) {
        $sql    .= " AND (v.name LIKE ? OR v.phone LIKE ? OR v.tag_number LIKE ? OR v.company LIKE ?) ";
        $types  .= 'ssss';
        $like    = '%' . $search . '%';
        $values[] = $like;
        $values[] = $like;
        $values[] = $like;
        $values[] = $like;
    }

    $sql .= " ORDER BY v.checkin_datetime DESC, v.id DESC";

    $stmt = $this->conn->prepare($sql);

    if ($types !== '') {
        $stmt->bind_param($types, ...$values);
    }

    $stmt->execute();
    return $stmt->get_result();
}

    public function createExpected($data)
{
    $sql = "
        INSERT INTO visitors 
            (name, phone, email, company, gender, address, purpose, items_declared,
             tag_number, host_id, department_id, status)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'expected')
    ";

    $stmt = $this->conn->prepare($sql);

    $stmt->bind_param(
        "ssssssssssi",
        $data['name'],
        $data['phone'],
        $data['email'],
        $data['company'],
        $data['gender'],
        $data['address'],
        $data['purpose'],
        $data['items_declared'],
        $data['tag_number'],
        $data['host_id'],
        $data['department_id']
    );

    if ($stmt->execute()) {
        return $this->conn->insert_id;
    }

    return false;
}
    
public function saveQrPath($id, $path)
{
    $stmt = $this->conn->prepare("UPDATE visitors SET qr_code_path = ? WHERE id = ?");
    $stmt->bind_param("si", $path, $id);
    return $stmt->execute();
  }
}