<?php
// ============================================================
// VISITOR GATE PASS MANAGEMENT SYSTEM
// CORE FUNCTIONS (settings + helpers)
// ============================================================

// Database connection
require_once __DIR__ . '/../config/db.php';

// ------------------------------------------------------------
// GET SETTING VALUE  (used by login page & others)
// ------------------------------------------------------------
function get_setting($key) {
    global $conn;
    $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = ? LIMIT 1");
    if (!$stmt) {
        return null;
    }
    $stmt->bind_param("s", $key);
    $stmt->execute();
    $stmt->bind_result($value);
    $stmt->fetch();
    $stmt->close();
    return $value;
}

// ------------------------------------------------------------
// setting_val() — generic settings helper (with default)
//   used by mailer, notifications, cron, etc.
// ------------------------------------------------------------
function setting_val($key, $default = '') {
    $val = get_setting($key);
    if ($val === null || $val === '') {
        return $default;
    }
    return trim($val);
}

// ------------------------------------------------------------
// SPLIT COMMA-SEPARATED EMAILS (shared helper)
// ------------------------------------------------------------
function split_emails($emails) {
    $list = [];
    foreach (explode(',', (string)$emails) as $e) {
        $e = trim($e);
        if ($e && filter_var($e, FILTER_VALIDATE_EMAIL)) {
            $list[] = $e;
        }
    }
    return array_unique($list);
}

// ------------------------------------------------------------
// SEND EMAIL (fallback – rarely used now, PHPMailer is primary)
// ------------------------------------------------------------
function send_email($to, $subject, $message) {
    $headers  = "MIME-Version: 1.0\r\n";
    $headers .= "Content-type:text/html;charset=UTF-8\r\n";

    $from_name = setting_val('vgp_email_from_name', 'Visitor Gate Pass');
    $from_addr = setting_val('vgp_email_from_address', 'noreply@example.com');
    $headers  .= "From: {$from_name} <{$from_addr}>\r\n";

    return mail($to, $subject, $message, $headers);
}

// ------------------------------------------------------------
// GET HOST INFO BY ID  (legacy – still available)
// ------------------------------------------------------------
function get_host($id) {
    global $conn;
    $stmt = $conn->prepare("SELECT name, email, department_id FROM hosts WHERE id = ? LIMIT 1");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $host   = $result->fetch_assoc();
    $stmt->close();
    return $host;
}

// ==================================================================
// FETCH EMPLOYEE INFO  (used by all visitor notifications)
// ==================================================================
function get_employee_email($id) {
    global $conn;
    $stmt = $conn->prepare("SELECT name, email FROM employees WHERE id = ? LIMIT 1");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();

    return [
        'name'  => $row['name']  ?? '',
        'email' => $row['email'] ?? ''
    ];
}

// ------------------------------------------------------------
// OVERSTAY CHECK
// ------------------------------------------------------------
function check_overstay($checkin_time) {
    $limit_hours   = (int) setting_val('overstay_limit_hours', 4);
    $limit_seconds = $limit_hours * 3600;
    return (time() - strtotime($checkin_time)) > $limit_seconds;
}

// ------------------------------------------------------------
// SECURE INPUT HELPER
// ------------------------------------------------------------
function sanitize($value) {
    global $conn;
    return htmlspecialchars(mysqli_real_escape_string($conn, trim((string)$value)));
}

// ------------------------------------------------------------
// LICENSE HELPER
// ------------------------------------------------------------
function setting($key, $default = null) {
    global $conn;
    $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = ? LIMIT 1");
    $stmt->bind_param('s', $key);
    $stmt->execute();
    $stmt->bind_result($value);
    if ($stmt->fetch()) {
        $stmt->close();
        return $value;
    }
    $stmt->close();
    return $default;
}

function setting_set($key, $value) {
    global $conn;
    $stmt = $conn->prepare("
        INSERT INTO settings (setting_key, setting_value)
        VALUES (?, ?)
        ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
    ");
    $stmt->bind_param('ss', $key, $value);
    $stmt->execute();
    $stmt->close();
}