<?php
/**
 * notify.php
 * ----------
 * All email notification helpers for the Visitor Gate Pass system.
 *
 * Uses:
 *   - sendMail(), setting_val(), split_emails() from mailer.php
 *   - DB connection from db.php
 */

require_once __DIR__ . '/mailer.php';
require_once __DIR__ . '/../config/db.php';

/* ---------------------------------------------
 * Helpers
 * ------------------------------------------ */

function get_department_name(int $department_id): string
{
    global $conn;

    $stmt = $conn->prepare(
        "SELECT name FROM departments WHERE id = ? LIMIT 1"
    );
    if (!$stmt) return '';

    $stmt->bind_param("i", $department_id);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();

    return $row['name'] ?? '';
}

function get_host_info(int $host_id): array
{
    global $conn;

    $stmt = $conn->prepare(
        "SELECT name, email FROM employees WHERE id = ? LIMIT 1"
    );
    if (!$stmt) {
        return ['name' => '', 'email' => ''];
    }

    $stmt->bind_param("i", $host_id);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();

    if (!$row) {
        return ['name' => '', 'email' => ''];
    }

    return [
        'name'  => $row['name'] ?? '',
        'email' => $row['email'] ?? '',
    ];
}

/* ========================================================
   1) VISITOR CHECK-IN
   - Recipients:
     * Host email (if set)
     * settings.notify_checkin_emails
======================================================== */
function notify_checkin(string $visitor_name, int $department_id, string $purpose, int $host_id): void
{
    $department = get_department_name($department_id);
    $host       = get_host_info($host_id);

    $recipients = [];

    if (!empty($host['email'])) {
        $recipients[] = $host['email'];
    }

    $extra = setting_val('notify_checkin_emails', '');
    $recipients = array_unique(array_merge($recipients, split_emails($extra)));

    if (empty($recipients)) {
        return;
    }

    $subject = 'Visitor Check-In Notification';

    $body = "A visitor has just checked in.\n\n" .
            "Visitor Name: {$visitor_name}\n" .
            "Department: {$department}\n" .
            "Host: {$host['name']}\n" .
            "Purpose: {$purpose}\n\n" .
            "-- Visitor Gate Pass System";

    @sendMail($recipients, $subject, $body);
}

/* ========================================================
   2) VISITOR CHECK-OUT
   - Recipients: settings.notify_checkout_emails
======================================================== */
function notify_checkout(string $visitor_name, string $checkout_time): void
{
    $extra      = setting_val('notify_checkout_emails', '');
    $recipients = split_emails($extra);

    if (empty($recipients)) {
        return;
    }

    $subject = 'Visitor Checked Out';

    $body = "A visitor has just checked out.\n\n" .
            "Visitor Name: {$visitor_name}\n" .
            "Checkout Time: {$checkout_time}\n\n" .
            "-- Visitor Gate Pass System";

    @sendMail($recipients, $subject, $body);
}

/* ========================================================
   3) EXPECTED VISITOR
   - Recipients: settings.notify_expected_emails
======================================================== */
function notify_expected_visitor(string $visitor_name, string $expected_date, string $host_name): void
{
    $extra      = setting_val('notify_expected_emails', '');
    $recipients = split_emails($extra);

    if (empty($recipients)) {
        return;
    }

    $subject = 'Expected Visitor Alert';

    $body = "An expected visitor has been registered.\n\n" .
            "Visitor Name: {$visitor_name}\n" .
            "Expected Date: {$expected_date}\n" .
            "Host: {$host_name}\n\n" .
            "-- Visitor Gate Pass System";

    @sendMail($recipients, $subject, $body);
}

/* ========================================================
   4) RETURNING VISITOR
   - Recipients: settings.notify_returning_emails
======================================================== */
function notify_returning_visitor(string $visitor_name, string $phone): void
{
    $extra      = setting_val('notify_returning_emails', '');
    $recipients = split_emails($extra);

    if (empty($recipients)) {
        return;
    }

    $subject = 'Returning Visitor Notification';

    $body = "A returning visitor has been detected.\n\n" .
            "Visitor Name: {$visitor_name}\n" .
            "Phone: {$phone}\n\n" .
            "-- Visitor Gate Pass System";

    @sendMail($recipients, $subject, $body);
}

/* ========================================================
   5) INDUCTION COMPLETED
   - Recipients: settings.notify_induction_emails
======================================================== */
function notify_induction_completed($name, $phone, $email = '', $company = '') {

    // Load system recipients from Settings
    $extra = setting_val('notify_induction_emails', '');
    $recipients = split_emails($extra);

    if (empty($recipients)) return; // No emails configured → do nothing.

    // Safe fallbacks
    $name    = $name    ?: 'Unknown Visitor';
    $phone   = $phone   ?: 'Not Provided';
    $email   = $email   ?: 'Not Provided';
    $company = $company ?: 'Not Provided';

    $subject = "Visitor Induction Completed";

    $body =
"Hello,

A visitor has just completed induction.

Name: {$name}
Phone: {$phone}
Email: {$email}
Organization: {$company}
Time: " . date('Y-m-d H:i:s') . "

-- Visitor Gate Pass System";

    @sendMail($recipients, $subject, $body); // safe
}

/* ========================================================
   6) OVERSTAY ALERT
   - Recipients: settings.overstay_alert_emails
======================================================== */
function notify_overstay(array $visitor, int $limit_minutes): void
{
    // NOTE: this uses overstay_alert_emails (already present in your settings)
    $emails = setting_val('overstay_alert_emails', '');
    $list   = split_emails($emails);

    if (empty($list)) {
        return;
    }

    $subject = 'Visitor Overstay Alert';

    $body  = "Hello,\n\n";
    $body .= "A visitor has overstayed beyond the allowed duration.\n\n";
    $body .= "Name: {$visitor['name']}\n";
    $body .= "Phone: {$visitor['phone']}\n";
    $body .= "Host: {$visitor['host_name']}\n";
    $body .= "Purpose: {$visitor['purpose']}\n";
    $body .= "Checked in: {$visitor['checkin_time']}\n";
    $body .= "Allowed duration: {$limit_minutes} minutes\n\n";
    $body .= "-- Visitor Gate Pass System";

    @sendMail($list, $subject, $body);
}