<?php
// ===============================================
// Visitor Gate Pass - Checkout Action (Tag Number Based)
// ===============================================
header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../includes/mailer.php';
require_once __DIR__ . '/../includes/functions.php';

// -----------------------------------------------
// 1️⃣ Validate tag number
// -----------------------------------------------
$tag = trim($_POST['tag_number'] ?? '');
if ($tag === '') {
    echo json_encode(['ok' => false, 'error' => 'Please enter a visitor tag number.']);
    exit;
}

// -----------------------------------------------
// 2️⃣ Fetch visitor by tag number
// -----------------------------------------------
$stmt = $conn->prepare("SELECT * FROM visitors WHERE tag_number=? LIMIT 1");
$stmt->bind_param("s", $tag);
$stmt->execute();
$visitor = $stmt->get_result()->fetch_assoc();

if (!$visitor) {
    echo json_encode(['ok' => false, 'error' => 'No visitor found with this tag number.']);
    exit;
}
if ($visitor['status'] !== 'checked_in') {
    echo json_encode(['ok' => false, 'error' => 'Visitor is not currently checked in.']);
    exit;
}

// -----------------------------------------------
// 3️⃣ Perform checkout update
// -----------------------------------------------
$update = $conn->prepare("UPDATE visitors SET status='checked_out', checkout_time=NOW() WHERE id=?");
$update->bind_param("i", $visitor['id']);
$update->execute();

// -----------------------------------------------
// 4️⃣ Sync induction table
// -----------------------------------------------
$updateInduction = $conn->prepare("UPDATE induction SET status='checked_out' WHERE phone=? OR email=?");
$updateInduction->bind_param("ss", $visitor['phone'], $visitor['email']);
$updateInduction->execute();

// -----------------------------------------------
// 5️⃣ Prepare notification data
// -----------------------------------------------
$name = $visitor['name'];
$phone = $visitor['phone'];
$email = $visitor['email'];
$host_id = (int)$visitor['host_id'];
$checkin = "{$visitor['checkin_date']} {$visitor['checkin_time']}";
$checkout = date('Y-m-d H:i:s');

// Build recipient list
$recipients = [];

// Host email
$host = get_employee_email($host_id);
if (!empty($host['email'])) {
    $recipients[] = $host['email'];
}

// System recipients from Settings (correct key!)
$extra = setting_val('notify_checkout_emails', '');
$more = split_emails($extra);

$recipients = array_unique(array_merge($recipients, $more));

// -----------------------------------------------
// 6️⃣ Send email notification
// -----------------------------------------------
if (!empty($recipients)) {
    $subject = "Visitor Checkout Notification";

    $body = "Hello,\n\nA visitor has checked out.\n\n"
          . "Name: {$name}\n"
          . "Phone: {$phone}\n"
          . "Email: {$email}\n"
          . "Tag Number: {$tag}\n"
          . "Check-In: {$checkin}\n"
          . "Check-Out: {$checkout}\n\n"
          . "-- Visitor Gate Pass System";

    // Suppress fatal errors to avoid breaking Ajax
    @sendMail($recipients, $subject, $body);
}

// -----------------------------------------------
// 7️⃣ Return JSON success
// -----------------------------------------------
echo json_encode(['ok' => true, 'message' => "{$name} (Tag: {$tag}) checked out successfully."]);
exit;